using System;
using System.Xml;
using gov.va.med.VBECS.Communication.Common;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents heartbeat response 
	/// message as defined in VistALink protocol.
	/// </summary>
	public sealed class VistALinkHeartbeatResponseMessage : VistALinkSimpleResponseMessage, IPingMessage
	{
		private Timeout _heartbeatRate;
	
		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "heartbeat";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_HEARTBEAT_RATE_RESPONSE_ATTRIBUTE_NAME = "rate";

		/// <summary>
		/// Minimal VistALink heartbeat response constructor. Heartbeat response message may be 
		/// created with heartbeat rate only - it should always have "success" response status 
		/// (despite specification does not explicitly define that). 
		/// </summary>
		public VistALinkHeartbeatResponseMessage( Timeout heartbeatRate ) 
			: this( heartbeatRate, ResponseStatus.Success ) {}

		/// <summary>
		/// Auxilliary constructor added to conform to the VistALink protocol specification. 
		/// Allows specifying VistALink heartbeat response message status. 
		/// </summary>
		/// <param name="heartbeatRate">
		///		Heartbeat rate - an interval in which client may remain 
		///		inactive before server disconnects.
		/// </param>
		/// <param name="responseStatus">Message response status.</param>
		public VistALinkHeartbeatResponseMessage( Timeout heartbeatRate, ResponseStatus responseStatus ) 
			: base( ResponseType, responseStatus )
		{
			if( heartbeatRate == null )
				throw( new ArgumentNullException( "heartbeatRate" ) );

			_heartbeatRate = heartbeatRate;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public VistALinkHeartbeatResponseMessage( XmlDocument sourceDoc ) 
			: base( ResponseType, sourceDoc )
		{
			_heartbeatRate = Timeout.ParseStringInSeconds( XmlUtility.ParseGetRequiredAttributeValue( XmlParseGetResponseElement( sourceDoc ), XMLCONSTS_HEARTBEAT_RATE_RESPONSE_ATTRIBUTE_NAME ) );
		}

		/// <summary>
		/// XML serialization method adding heartbeat 
		/// rate to the base class response attributes. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseAttributes( XmlWriter writer )
		{
			base.WriteXmlResponseAttributes( writer );
			writer.WriteAttributeString( XMLCONSTS_HEARTBEAT_RATE_RESPONSE_ATTRIBUTE_NAME, _heartbeatRate.ToStringSeconds() );
		}

		/// <summary>
		/// The heartbeat rate specifies maximum period of inactivity tolerated by 
		/// VistALink listener. After this period is expired, heartbeat request message 
		/// must be sent by client (server will respond with a heartbeat response message),
		/// otherwise connection will be dropped. 
		/// Basically, heartbeat rate is timeout (or interval). It was named rate 
		/// because of consistency issues. 
		/// </summary>
		public Timeout HeartbeatRate
		{
			get
			{
				return _heartbeatRate;
			}
		}
	}
}
